%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% File: randomizeBabbles.m
%
% Randomizes a set of babbles for 
%   bootstrapping, so the structure
%   of each chat file is kept the same
%   (i.e. the syllables per babble).
%
% Inputs:
%   * babbles: The babbles to randomize
%   * r_level: The level to randomize at.
%       1 - randomize within each line
%       2 - randomize within each babble
%       3 - randomize across all babbles
%
% Outputs: 
%   * r_babbles: The randomized babbles.
%
% Usage: r_babbles = randomizeBabbles(babbles, 2);
%
% Author: Doug Bemis
% Date: 11/27/11
%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function r_babbles = randomizeBabbles(babbles,use_known_transitions)

% Always randomize at the level of the session for now
r_level = 2;

% Store the syllables to randomize here
r_syllables = [];
r_babbles = babbles;

% The babbles to randomize should align with those
%   used in tabulateBabble.
% So, only complete, non-ambiguous parses for now.

% Go through each babble, which is a session for the child
for s = 1:length(babbles)
    curr_session = babbles{s};

    % Go through each utterance
    for u = 1:length(curr_session)
        curr_utterance = curr_session{u};
        
        % This will return the syllables of that utterance
        syllables = getSyllablesFromParse(curr_utterance);
        if ~isempty(syllables)
            
            % If we got some, accumulate them in a large array
            r_syllables(end+1:end+length(syllables)) = syllables;

            % If randomizing per parse, do here
            if r_level == 1
                replaced = replaceSyllables({{babbles{s}{u}}}, r_syllables, use_known_transitions);
                r_babbles{s}{u} = replaced{1}{1};
                r_syllables = [];
            end
        end
    end
    
    % If randomizing per session, do here
    if r_level == 2
        
        % Shuffle them around
        replaced = replaceSyllables({babbles{s}}, r_syllables, use_known_transitions);
        
        % And replace in our return struct
        r_babbles{s} = replaced{1};
        
        % And clear the accumulated syllables
        r_syllables = [];
    end
end

% Finally, if randomizing at the top levell
if r_level == 3
    r_babbles = replaceSyllables(babbles, r_syllables, use_known_transitions);
end

% Helper to replace the syllables
function babbles = replaceSyllables(babbles, r_syllables, use_known_transitions)

if isempty(r_syllables)
    return;
end

% Comment this in to just randomize
% First, randomize our syllables to use
if use_known_transitions

    % This will only use transitions that are seen in the rest of the data.
    load('all_transitions.mat');

    % Randomize unti we only have known transitions
    is_good = 0;
    ctr_3 = 0;
    while ~is_good && ctr_3 < 10
%         ctr_3 = ctr_3+1;
        
        is_good = 1;
        
        % Set up some temp arrays to hold all the syllables
        %   The final order is going to end up in tmp_2
        tmp = r_syllables;
        tmp_2 = zeros(size(tmp));
        r_ind = randperm(length(tmp));
        
        % Get it started
        tmp_2(1) = tmp(r_ind(1));
        tmp(r_ind(1)) = -1;
        ctr = 2;
        ctr_2 = 0;
        
        % Go through all of the syllables
        while ctr <= length(tmp)
            
            % Set to a known transition
            for j = randperm(length(tmp))
                if tmp(j) > 0 && all_transitions(tmp_2(ctr-1),tmp(j)) == 1
                    tmp_2(ctr) = tmp(j);
                    tmp(j) = -1;
                    break;
                end
            end
            
            % We failed if here, so give it another shot and reset
            if tmp_2(ctr) == 0
                ctr_2 = ctr_2 +1;
                tmp = r_syllables;
                ctr = ctr-1;
                
                % Only one chance at this, so signal failure
                if ctr_2 == 2
                    is_good = 0;
                    break;
                end
            end
            ctr = ctr+1;
        end
        if is_good
            r_syllables = tmp_2;
        else
            % Failure handling here...
        end

        % Check
        if is_good
            for i = 1:length(r_syllables)-1
                if all_transitions(r_syllables(i),r_syllables(i+1)) == 0
                    is_good = 0;
                    error('Bad somehow...');
                    break;
                end
            end
        end
    end
else
    r_syllables = r_syllables(randperm(length(r_syllables)));    
end

% Then, go through each session...
ctr = 0;
for s = 1:length(babbles)
    curr_session = babbles{s};

    % And each utterrance...
    for u = 1:length(curr_session)
        curr_utterance = curr_session{u};

        % See if we got the syllables
        syllables = getSyllablesFromParse(curr_utterance);

        % If so...
        if ~isempty(syllables)

            % Get the next set of syllables
            r_ind = ctr+1:ctr+length(syllables);

            % Replace the babble
            babbles{s}{u}{4}{1}{1} = r_syllables(r_ind);

            % And advance the counter
            ctr = ctr+length(syllables);
        end
    end
end

% Check
if ctr ~= length(r_syllables)
    error('Wrong number of syllables replaced. Exiting...');
end

