%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% File: plotTrends.m
%
% Creates a scatterplot from a 
%   set of results for each syallable 
%   or transition, for each session 
%   for which it is above threshold.
%       * An average per session is 
%       added afterwards.
%
% NOTE: The sessions are relative to 
%   when the syllable or transition 
%   first appeared and so do not
%   correspond to a specific file.
%
% Inputs:
%   * trend_type: The stat to plot.
%       - SylEdgeFreq: Frequency per session
%           that a syllable occurred at an edge location.
%   * results: The output from analyzeChild to plot.
%   * syl_baseline: The % of total syllables from 
%       that day that the syllable had to be to count
%       as not chance.
%   * ingore_prior_syllables: 1 to not count syllables
%       that are above threshold on the first day.
%   * min_syl_freq: The minimum number of occurrences of a
%       syllable in a session to count that syllable
%   * min_sess_syl: The minimum number of syllables in a 
%       session to plot it.
%   * save_name: If present will save the plot to this filename
%   * display_type:
%       1: Scatter plot with means overlaid
%       2: Means with standard error bars
%       3: Means only
%
% Outputs: None, but creates the plot, and possibly saves it
%
% Usage: plotTrends('SylEdgeFreq', results, 0.01, 1, ...
%   5, 10, 'Frequency_Plot.jpg');
%
% Author: Doug Bemis
% Date: 11/27/11
%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [means stderrs all_values age_means age_stderrs norms norm_errs all_norms] = plotTrends(results, trend_type, alignment, syl_baseline, tr_baseline, ...
    ignore_prior_syllables, min_syl_freq, min_sess_syl, min_perc_babble, min_perc_parse, min_age, num_boots, image_file, ...
    values_file, display_type, babbles, files, display_boots, boots_image_file, use_known_transitions)

% Default to not save
if ~exist('image_file','var')
    image_file = '';
end
if ~exist('values_file','var')
    values_file = '';
end

% Default to display a scatter plot
if ~exist('display_type','var')
    display_type = 1;
end

if ~exist('display_boots','var')
    display_boots = 0;
end
if ~exist('boots_image_file','var')
    boots_image_file = '';
end
if ~exist('alignment','var')
    alignment = 0;
end

% Default to randomization over all transitions
if ~exist('use_known_transitions','var')
    use_known_transitions = 0;
end

% Find out which are over threshold and not noise, where
%   the baselines are % of total syllables in the session.
syl_above_baseline = squeeze(results.syl_freq(1,:,:)) ./ ...
    repmat(results.total_syl,size(results.syl_freq,2),1) > syl_baseline;

% Do the same for transitions

% This will give a percentage cutoff
if tr_baseline < 1
    tmp = repmat(results.total_syl,[size(results.tr_freq,1),1,size(results.tr_freq,2)]);
    tmp = permute(tmp,[1,3,2]);
    tr_above_baseline = (results.tr_freq*2 ./ tmp) > tr_baseline;
else
    tr_above_baseline = results.tr_freq > tr_baseline; 
end

% Record all the transistions
% global all_t;
% for i = 1:size(tr_above_baseline,1)
%     for j = 1:size(tr_above_baseline,2)
%         
%         if sum(tr_above_baseline(i,j,:)) > 0
%             if i == 1 && j == 15
%                 disp('asf');
%             end
%             all_t(i,j) = 1;
%         end
%     end
% end

% If normalizing, do it now
if num_boots > 0
    
    % Set the display type
    if display_boots
        b_dis_type = 3;
    else
        b_dis_type = 0;
    end
    
    % Get the means
    max_norm_session = length(files);
    boot_means = {};
    for b = 1:num_boots
        disp(['Normalizing, boot ' num2str(b) '...']);
        
        % Randomize the syllables
        r_babbles = randomizeBabbles(babbles,use_known_transitions);

        % These are still the same, just reconstruct
        good_parses = [];
        non_babbles = {};
        for f = 1:length(files)
            good_parses(f) = results.perc_parsed(f)*length(babbles{f}); %#ok<AGROW>
            if results.perc_babbling(f) > 0
                non_babbles{f} = cell(1,round(length(babbles{f})/results.perc_babbling(f) - length(babbles{f}))); %#ok<AGROW>
            else
                non_babbles{f} = {{}};  %#ok<AGROW>
            end
        end
        
        % Reanalyze the shuffled babbles
        % NOTE: Need to unhardcode if we ever add another database.
        %   though should probably be stored with the babbles, etc.
        r_results = analyzeChild('', '', '', 0, '', r_babbles, files, good_parses, non_babbles);
%         r_results.file_ages = results.file_ages;
        
        % Get the new means for this bootstrap
        boot_means{b} = plotTrends(r_results, trend_type, alignment, syl_baseline, tr_baseline, ignore_prior_syllables,...
            min_syl_freq, min_sess_syl,min_perc_babble, min_perc_parse, min_age, 0, '','',b_dis_type); %#ok<AGROW>
        if length(boot_means{b}) < max_norm_session
            max_norm_session = length(boot_means{b});
        end
    end
    
    % Reorganize and calculate
    all_norms = zeros(num_boots,max_norm_session);
    for b = 1:num_boots
        all_norms(b,:) = boot_means{b}(1:max_norm_session);
    end
    if num_boots > 1
        norms = mean(all_norms);
    else
        norms = all_norms;
    end
    norm_errs = std(all_norms) / sqrt(num_boots);
    
    % Save the figure, if we want it
    if display_boots && ~isempty(boots_image_file)
        saveas(gcf,boots_image_file);
    end
    
    % Make a new figure for the next display
    if display_boots && display_type > 0
        figure;
    end
else
    norms = [];
    norm_errs = [];
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Syllable plots
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Start the output file
num_files = size(results.syl_freq, 3);
if ~isempty(values_file)
    fid = fopen(values_file,'w');

    % Print the header
    fprintf(fid,'Syllable');
    for f = 1:num_files
        fprintf(fid,['\t' num2str(f)]);
    end
    fprintf(fid,'\n');
end

% Get the syllable stats
all_values = {};
all_ages = {};
syl_ages = [];
age_means = [];
age_stderrs = [];
for s = 1:size(results.syllables,2)
    
    % Ignore the unrepresented ones
    if isempty(results.syllables{1,s})
        continue;
    end

    % Don't count if not ever above chance
    if ~any(syl_above_baseline(s,:) > 0)
        continue;
    end
    
    % Don't count any that were above baseline in the first session
    % TODO: Only for above baseline, or for any appearance in the first
    % session.
    if ignore_prior_syllables && syl_above_baseline(s,1)
        continue;
    end

    % Find the session #s that count for this syllable
    % NOTE: Functionalize and take outside switch
    ind = getGoodSessions(s, syl_above_baseline, results, min_syl_freq, min_perc_babble, min_perc_parse);
    if length(ind) < 1
        continue;
    end

    % Calculate the stats
    switch(trend_type)
                
        % These are the nine types of transition placements a syllable can
        % be in.
        case 'None_None'
             syl_values = squeeze((results.syl_pos_freq(1,s,ind,1,1) ./ results.syl_freq(1,s,ind)));            
            
        case 'None_Same'
             syl_values = squeeze((results.syl_pos_freq(1,s,ind,1,2) ./ results.syl_freq(1,s,ind)));            
            
        case 'None_Diff'
             syl_values = squeeze((results.syl_pos_freq(1,s,ind,1,3) ./ results.syl_freq(1,s,ind)));            
            
        case 'Same_None'
             syl_values = squeeze((results.syl_pos_freq(1,s,ind,2,1) ./ results.syl_freq(1,s,ind)));            
            
        case 'Same_Same'
             syl_values = squeeze((results.syl_pos_freq(1,s,ind,2,2) ./ results.syl_freq(1,s,ind)));            
            
        case 'Same_Diff'
             syl_values = squeeze((results.syl_pos_freq(1,s,ind,2,3) ./ results.syl_freq(1,s,ind)));            
            
        case 'Diff_None'
             syl_values = squeeze((results.syl_pos_freq(1,s,ind,3,1) ./ results.syl_freq(1,s,ind)));            
            
        case 'Diff_Same'
             syl_values = squeeze((results.syl_pos_freq(1,s,ind,3,2) ./ results.syl_freq(1,s,ind)));            
            
        case 'Diff_Diff'
             syl_values = squeeze((results.syl_pos_freq(1,s,ind,3,3) ./ results.syl_freq(1,s,ind)));            
            
        % This plots all of the edges, including surrounded by the same
        % syllable. So, this is only not when between two different
        % syllables
        case 'SylEdgeFreq'                
            
             % This will be any occurrence that doesn't a different
             % syllable on both sides.
             syl_values = squeeze((1-(results.syl_pos_freq(1,s,ind,3,3) ./ results.syl_freq(1,s,ind))));
             
             % Get the ages
             if isfield(results,'file_ages')
                 syl_ages = results.file_ages(ind);
             end
             
        % This is on the front of the utterrence 
        case 'Fr_Edge'                
            
            % Get the values
            freq = squeeze(results.syl_pos_freq(1,s,ind,1,2)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,1,3));
            syl_values = squeeze((freq ./ squeeze(results.syl_freq(1,s,ind))));
                        
        % This is on the end of the utterrence 
        case 'End_Edge'                
            
            % Get the values
            freq = squeeze(results.syl_pos_freq(1,s,ind,2,1)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,3,1));
            syl_values = squeeze((freq ./ squeeze(results.syl_freq(1,s,ind))));
                        
        % These are all syllables with one other transition
        case 'One_Tr'                
            
            % Get the values
            freq = squeeze(results.syl_pos_freq(1,s,ind,2,1)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,3,1)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,1,2)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,1,3));
            syl_values = squeeze((freq ./ squeeze(results.syl_freq(1,s,ind))));
                        
        % These are all syllables with one other transition
        case 'Two_Tr'                
            
            % Get the values
            freq = squeeze(results.syl_pos_freq(1,s,ind,2,2)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,3,2)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,2,3));
            syl_values = squeeze((freq ./ squeeze(results.syl_freq(1,s,ind))));
                        
        % This is where the difference comes on the right 
        case 'Diff_Right'                
            
            % Get the values
            freq = squeeze(results.syl_pos_freq(1,s,ind,1,3)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,2,3));
            syl_values = squeeze((freq ./ squeeze(results.syl_freq(1,s,ind))));
                        
        % This is where the difference comes on the left 
        case 'Diff_Left'                
            
            % Get the values
            freq = squeeze(results.syl_pos_freq(1,s,ind,3,1)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,3,2));
            syl_values = squeeze((freq ./ squeeze(results.syl_freq(1,s,ind))));
                        
        % Anytime there is a different syllable
        case 'Diff'                
            
            % Get the values
            freq = squeeze(results.syl_pos_freq(1,s,ind,1,3)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,2,3)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,3,3)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,3,2)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,3,1));
            syl_values = squeeze((freq ./ squeeze(results.syl_freq(1,s,ind))));
                        
        % One diff syllable
        case 'One_Diff'                
            
            % Get the values
            freq = squeeze(results.syl_pos_freq(1,s,ind,1,3)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,2,3)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,3,2)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,3,1));
            syl_values = squeeze((freq ./ squeeze(results.syl_freq(1,s,ind))));

        % Alone or only next to itself
        case 'Same'                
            
            % Get the values
            freq = squeeze(results.syl_pos_freq(1,s,ind,1,1)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,1,2)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,2,1)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,2,2));
            syl_values = squeeze((freq ./ squeeze(results.syl_freq(1,s,ind))));

        % Reduplication only 
        case 'Redup'                
            
            % Get the values
            freq = squeeze(results.syl_pos_freq(1,s,ind,1,2)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,2,1)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,2,2));
            syl_values = squeeze((freq ./ squeeze(results.syl_freq(1,s,ind))));
                        
        % Next to any double
        case 'SylDoubleFreq'                
            
            % Get the values
            freq = squeeze(results.syl_pos_freq(1,s,ind,2,1)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,2,2)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,2,3)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,1,2)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,3,2));
            syl_values = squeeze((freq ./ squeeze(results.syl_freq(1,s,ind))));
            
        case 'All'                
            
            % Get the values
            freq = squeeze(results.syl_pos_freq(1,s,ind,1,1)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,1,2)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,1,3)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,2,1)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,2,2)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,2,3)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,3,1)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,3,2)) + ...
                squeeze(results.syl_pos_freq(1,s,ind,3,3));
            syl_values = squeeze((freq ./ squeeze(results.syl_freq(1,s,ind))));
            
        % When the syllable first occurred
        case 'SylFirstOcc'
            syl_values = zeros(length(ind),1);
            syl_values(1) = 1;
            
        % Transitions per syllable
        case 'SylTrans'
            if s > size(results.tr_freq,1) || s > size(results.tr_freq,2)
                continue;
            end
            num_sec_syl = size(results.tr_freq,2);
            
            % Get the number of transitions for each accepted session,
            %   not counting doubles
            syl_values = squeeze(sum(results.tr_freq(s,[1:s-1 s+1:num_sec_syl],ind))) + ...
                squeeze(sum(results.tr_freq(:,s,ind)));
            
        % Transitions per syllable, not including duplications, per
        % frequency
        case 'SylTransProp'
            if s > size(results.tr_freq,1) || s > size(results.tr_freq,2)
                continue;
            end
            num_sec_syl = size(results.tr_freq,2);
            
            % Get the number of transitions for each accepted session,
            %   not counting doubles
            syl_values = (squeeze(sum(results.tr_freq(s,[1:s-1 s+1:num_sec_syl],ind))) + squeeze(sum(results.tr_freq(:,s,ind))))...
                ./ squeeze(results.syl_freq(1,s,ind));
            
        % Count the frequency of distinct transitions to syllable
        % occurrences
        case 'SylTransTypes'
            if s > size(results.tr_freq,1) || s > size(results.tr_freq,2)
                continue;
            end
            num_sec_syl = size(results.tr_freq,2);
            syl_values = squeeze(sum(results.tr_freq(s,[1:s-1 s+1:num_sec_syl],ind) > tr_baseline)) +...
                squeeze(sum(results.tr_freq(:,s,ind) > tr_baseline)) ./ squeeze(results.syl_freq(1,s,ind));          
             syl_values(isnan(syl_values)) = 1;

        % Count the cumiulative distinct transitions per syllable 
        case 'CumSylTransTypes'
            if s > size(results.tr_freq,1) || s > size(results.tr_freq,2) 
                continue;
            end
            
            % Have to do the slow way
            syl_values = [];
            attested_left = [];
            attested_right = [];
            for i = ind
                new_left = find(results.tr_freq(s,:,i) > tr_baseline);
                new_right = find(results.tr_freq(:,s,i) > tr_baseline)';
                for n = new_left
                    if ~any(attested_left == n)
                        attested_left(end+1) = n;  %#ok<AGROW>
                    end
                end
                for n = new_right
                    if ~any(attested_right == n)
                        attested_right(end+1) = n;  %#ok<AGROW>
                    end
                end
                syl_values(end+1) = length(attested_left)+length(attested_right);  %#ok<AGROW>
                
                % Find total possible
                tot_syl = sum(sum(results.syl_freq(:,:,1:i),3)>0);
                tot_tr = 2*tot_syl-1;
                syl_values(end) = syl_values(end) / tot_tr;
            end
            syl_values = syl_values';

        % Count the gain of distinct transitions to syllable occurrences
        case 'CumSylTransTypesDiff'
            if s > size(results.tr_freq,1) || s > size(results.tr_freq,2) || length(ind) == 1
                continue;
            end
            
            % Have to do the slow way
            tot_avail = [];
            syl_values = [];
            attested_left = [];
            attested_right = [];
            for i = ind
   
                new_left = find(results.tr_freq(s,:,i) > tr_baseline);
                new_right = find(results.tr_freq(:,s,i) > tr_baseline)';
                for n = new_left
                    if ~any(attested_left == n)
                        attested_left(end+1) = n;  %#ok<AGROW>
                    end
                end
                for n = new_right
                    if ~any(attested_right == n)
                        attested_right(end+1) = n;  %#ok<AGROW>
                    end
                end
                syl_values(end+1) = length(attested_left)+length(attested_right);  %#ok<AGROW>
                
                % Find total possible
                if i > ind(1)
                    tot_syl = sum(sum(results.syl_freq(:,:,1:i),3)>0);
                    tot_tr = 2*tot_syl-1; 
                    tot_avail(end+1) = tot_tr - syl_values(end-1); %#ok<AGROW>
                end
            end
            
            % Now, reset to show frequencies that increase came in
            ind = ind(2:end);
%             syl_values = diff(syl_values)' ./ squeeze(results.syl_freq(1,s,ind));

            % For total proportion of available
            syl_values = diff(syl_values)' ./ tot_avail';


            % For total #
%             syl_values = diff(syl_values)';
            
        % This is essentially wrong, it averages the reversed onsets
        % relative to syllables first which weights the transitions weirdly
        case 'RevTransOnset_Old'
            if s > min(size(tr_above_baseline,1),size(tr_above_baseline,2))
                continue;
            end
            syl_values = zeros(1,num_files);
            for s2 = 1:size(results.syllables,2)
                if s2 > min(size(tr_above_baseline,1),size(tr_above_baseline,2)) || s == s2
                    continue;
                end
                tr_ind = find(tr_above_baseline(s,s2,:) > 0);
                if ~isempty(tr_ind)
                    rev_tr_ind = find(tr_above_baseline(s2,s,:) > 0);
                    if ~isempty(rev_tr_ind)
                        diff_on = abs(rev_tr_ind(1) - tr_ind(1))+1;
                        syl_values(diff_on) = syl_values(diff_on) + 1;  
                    end
                end
            end
            
        % This gets the distribution of reverse onsets
        case 'RevTransOnset'
            if s > min(size(tr_above_baseline,1),size(tr_above_baseline,2))
                continue;
            end
            
            % This signals that we're splitting by transition and not
            % syllable...
            syl_values = [];
            for s2 = 1:size(results.syllables,2)
                if s2 > min(size(tr_above_baseline,1),size(tr_above_baseline,2)) || s == s2
                    continue;
                end
                tr_ind = find(tr_above_baseline(s,s2,:) > 0 );
                if ~isempty(tr_ind)
                    rev_tr_ind = find(tr_above_baseline(s2,s,:) > 0);
                    if ~isempty(rev_tr_ind)
                        
                        % Get the onset differences
                        diff_on = abs(rev_tr_ind(1) - tr_ind(1))+1;

                        % Store each one as a value
                        for i = 1:max(diff_on,length(all_values))
                            if i > length(all_values)
                                if i == 1
                                    all_values{i} = []; %#ok<AGROW>
                                else
                                    all_values{i} = zeros(1,length(all_values{i-1})-1); %#ok<AGROW>
                                end
                            end
                            if i == diff_on
                                all_values{i}(end+1) = 1; %#ok<AGROW>
                            else
                                all_values{i}(end+1) = 0; %#ok<AGROW>
                            end
                        end
                    end
                end
            end
            
        % This calculates absolute lag (in the third index - first two are
        %   onset of first syllable and onset of second syllable)
        case 'RevTransLag'
            if s > min(size(tr_above_baseline,1),size(tr_above_baseline,2))
                continue;
            end
            
            % This signals that we're splitting by transition and not
            % syllable...
            syl_values = [];
            for s2 = 1:size(results.syllables,2)
                if s2 > min(size(tr_above_baseline,1),size(tr_above_baseline,2)) || s == s2
                    continue;
                end
                tr_ind = find(tr_above_baseline(s,s2,:) > 0 );
                if ~isempty(tr_ind)
                    rev_tr_ind = find(tr_above_baseline(s2,s,:) > 0);
                    if ~isempty(rev_tr_ind) && rev_tr_ind(1) >= tr_ind(1)
                        
                        if rev_tr_ind(1) == tr_ind(1) 
                            if s > s2
                                continue;
                            end
                        end
                        if sum(results.tr_freq(s,s2,:)) < 1 || sum(results.tr_freq(s2,s,:)) < 1
                            continue;
                        end
                        
                        % Get the lag, counting by syllable occurrence
                        % sessions
                        onset_1 = find(ind == tr_ind(1),1);
                        onset_2 = find(ind == rev_tr_ind(1),1);
                        if isempty(onset_1) || isempty(onset_2)
                            continue;
                        end
            
                        % This will just take the absolue position
%                         onset_1 = tr_ind(1);
%                         onset_2 = rev_tr_ind(1);
                        diff_on = abs(onset_2 - onset_1);

                        % Need to store each as its own value here...
                        for i = 1:20        % NOTE: This value is arbitrary but needs to be higher than the minimum syllable sessions
                            if i > length(all_values)
                                if i == 1
                                    all_values{i} = []; %#ok<AGROW>
                                else
                                    all_values{i} = zeros(1,length(all_values{i-1})-1); %#ok<AGROW>
                                end
                            end
                            
                            % We'll just set them all to the first value,
                            % which will then be averaged and the rest to
                            % 1, to avoid dividing by 0.
                            if i == 1
                                all_values{i}(end+1) = onset_1; %#ok<AGROW>
                            elseif i == 2
                                all_values{i}(end+1) = onset_2; %#ok<AGROW>
                            elseif i == 3
                                all_values{i}(end+1) = diff_on; %#ok<AGROW>
                            else
                                all_values{i}(end+1) = 1; %#ok<AGROW>
                            end
                        end
                    end
                end
            end
            
        % This counts the proportion of reverse transitions relative to
        % their overall frequency, relative to the onset of the first
        % syllable.
        case 'RevTransProp'
            if s > min(size(tr_above_baseline,1),size(tr_above_baseline,2))
                continue;
            end
            
            % This signals that we're splitting by transition and not
            % syllable...
            syl_values = [];
            for s2 = 1:size(results.syllables,2)
                if s2 > min(size(tr_above_baseline,1),size(tr_above_baseline,2)) || s == s2
                    continue;
                end
                tr_ind = find(tr_above_baseline(s,s2,:) > 0 );
                if ~isempty(tr_ind)
                    rev_tr_ind = find(tr_above_baseline(s2,s,:) > 0);
                    if ~isempty(rev_tr_ind)
                        s_val = zeros(1,num_files);
                        if tr_ind(1) <= rev_tr_ind(1)
                            s_val(rev_tr_ind-tr_ind(1)+1) = squeeze(results.tr_freq(s2,s,rev_tr_ind)) / sum(results.tr_freq(s2,s,rev_tr_ind));
                        else
                            continue;
                        end

                        % Need to store each as its own value here...
                        for i = 1:length(s_val)        % NOTE: This value is arbitrary but needs to be higher than the minimum syllable sessions
                            if i > length(all_values)
                                all_values{i} = []; %#ok<AGROW>
                            end
                            all_values{i}(end+1) = s_val(i); %#ok<AGROW>
                        end
                    end
                end
            end

        % Just count when we're onsetting
        case 'SylOnset'
            syl_values = zeros(num_files,1);
            syl_values(find(syl_above_baseline(s,:) > 0,1)) = 1;

        % Sum new syllables
        case 'NewSyl'
            syl_values = zeros(num_files,1);
            syl_values(find(syl_above_baseline(s,:) > 0,1)) = 1;
            
        case 'UsedSyl'
            syl_values = zeros(num_files,1);
            syl_values(1) = 1;
            
        case 'UsedTr'
            syl_values = zeros(num_files,1);
            if s > min(size(tr_above_baseline,1),size(tr_above_baseline,2))
                continue;
            end
            for s2 = 1:size(results.syllables,2)
                if s2 > min(size(tr_above_baseline,1),size(tr_above_baseline,2)) 
                    continue;
                end
                if sum(tr_above_baseline(s,s2,:)) > 0
                    syl_values(1) = syl_values(1)+1;
                end
            end            
            
        % This is old and probably not to be used
        case 'RevTransOnsetSing'
            if s > min(size(tr_above_baseline,1),size(tr_above_baseline,2))
                continue;
            end
            syl_values = zeros(num_files,1);
            for s2 = 1:size(results.syllables,2)
                if s2 > min(size(tr_above_baseline,1),size(tr_above_baseline,2)) || s == s2
                    continue;
                end
                tr_ind = find(tr_above_baseline(s,s2,:) > 0 );
                if ~isempty(tr_ind)
                    rev_tr_ind = find(tr_above_baseline(s2,s,:) > 0);
                    if ~isempty(rev_tr_ind)
                        if rev_tr_ind(1) >= tr_ind(1)
                            syl_values(2) = syl_values(2) +1;
                            syl_values(1) = syl_values(1) + rev_tr_ind(1) - tr_ind(1); 
                        end
                    end
                end
            end
            if syl_values(2) > 0
                syl_values(1) = syl_values(1) / syl_values(2);
    %             syl_values(2) = 0;
            end        
            
        % Basic count of utterance length per session
        case 'Utt_Len'
            syl_values = results.utt_len(~isnan(results.utt_len));
            
        % Basic count of syllables per session
        case 'Syl_Num'
            syl_values = squeeze(sum(results.syl_freq));
            
        % Basic count of transitions per session
        case 'Tr_Num'
            syl_values = squeeze(sum(sum(results.tr_freq)));
            
        % Basic count distinct syllables per session
        case 'Diff_Syl_Num'
            syl_values = squeeze(sum(results.syl_freq > 0));
            
        % Basic count of distinct transitions per session
        case 'Diff_Tr_Num'
            syl_values = squeeze(sum(sum(results.tr_freq > 0)));
            
        % Basic count of all syllables 
        case 'Tot_Syl_Num'
            syl_values = zeros(num_files,1);
            syl_values(1) = sum(sum(results.syl_freq));
            
        % Basic count of all transitions
        case 'Tot_Tr_Num'
            syl_values = zeros(num_files,1);
            syl_values(1) = sum(sum(sum(results.tr_freq)));
            
        % Basic count of all different syllables
        case 'Tot_Diff_Syl_Num'
            syl_values = zeros(num_files,1);
            syl_values(1) = sum(sum(results.syl_freq,3) > 0);
            
        % Basic count of all different transitions
        case 'Tot_Diff_Tr_Num'
            syl_values = zeros(num_files,1);
            syl_values(1) = sum(sum(sum(results.tr_freq,3) > 0));
            
    end
    
    % Realign here, if necessary
    if alignment > 1 && ~isempty(syl_values)
    
        % Find the useable sessions
        if alignment == 4
            good_sess = find(squeeze(results.perc_babbling > min_perc_babble) & ...
                squeeze(results.perc_parsed > min_perc_parse) & ...
                squeeze(results.file_ages' > min_age));
            
        else
            good_sess = find(squeeze(results.perc_babbling > min_perc_babble) & ...
                squeeze(results.perc_parsed > min_perc_parse));

            % If we're back aligning, reverse
            if alignment == 3
                tmp = [];
                for i = length(good_sess):-1:1
                    tmp(end+1) = good_sess(i);             %#ok<AGROW>
                end
                good_sess = tmp;
            end
        end        
        
        % Move our syllable values into session space
        tmp = syl_values;
        syl_values = ones(length(good_sess),1)*NaN;
        if ~isempty(syl_ages)
            tmp_ages = syl_ages;
            syl_ages = ones(length(good_sess),1)*NaN;
        end
        for i = 1:length(ind)
            syl_values(good_sess == ind(i)) = tmp(i);
            if ~isempty(syl_ages)
                syl_ages(good_sess == ind(i)) = tmp_ages(i);
            end
        end
        
        % This will bunch into twos
%         syl_values = ones(ceil(length(good_sess)/2),1)*NaN;
%         for i = 1:length(ind)
%             new_ind = ceil(find(good_sess == ind(i),1)/2);
%             if isnan(syl_values(new_ind))
%                 syl_values(new_ind) = tmp(i);  
%             else
%                 syl_values(new_ind) = (tmp(i)+syl_values(new_ind))/2;  
%             end
%         end
    end
    
%     % Normalize here
%     if num_boots > 0
%         if length(syl_values) > max_norm_session
%             syl_values = syl_values(1:max_norm_session) - norms';
% %             syl_values = norms';
%         else
%             syl_values = syl_values - norms(1:length(syl_values))';
% %             syl_values = norms(1:length(syl_values))';
%         end
%     end
% 
%     % Plot if called for
%     if display_type == 1
%         scatter(1:length(syl_values),syl_values);
%     end
%     hold on;
% 
%     % Print if necessary
%     if ~isempty(values_file)
%         fprintf(fid,num2str(s));
%         for v = 1:length(syl_values)
%             fprintf(fid,['\t' num2str(syl_values(v))]);
%         end
%         fprintf(fid,'\n');
%     end
% 
    % And store for means, stderr
    if ~isempty(syl_values)
        for i = 1:length(syl_values)
            if i > length(all_values)
                all_values{i} = []; %#ok<AGROW>
                if ~isempty(syl_ages)
                    all_ages{i} = []; %#ok<AGROW>
                end
            end
            all_values{i}(end+1) = syl_values(i); %#ok<AGROW>
            if ~isempty(syl_ages)
                all_ages{i}(end+1) = syl_ages(i);  %#ok<AGROW>
            end
        end
    end
end

% Check for empty data
i = 1;
while i <= length(all_values)
    if sum(isnan(all_values{i})) >= length(all_values{i}) &&...
            length(all_values{i}) > 1
        tmp = all_values;
        all_values = {};
        for j = 1:length(tmp)
            if j < i
                all_values{j} = tmp{j};
            elseif j > i
                all_values{j-1} = tmp{j};
            end
        end
        if ~isempty(all_ages)
            tmp_ages = all_ages;
            all_ages = {};
            for j = 1:length(tmp_ages)
                if j < i
                    all_ages{j} = tmp_ages{j};
                elseif j > i
                    all_ages{j-1} = tmp_ages{j};
                end
            end
        end            
        i = i-1;
    end
    i = i+1;
end

% Set the max session to use
for max_session = length(all_values):-1:1
    if length(all_values{max_session}) >= min_sess_syl
        break;
    end
end

% To get sum as opposed to mean...
if strcmp(trend_type,'NewSyl') || strcmp(trend_type,'UsedSyl') || strcmp(trend_type,'UsedTr')
    for s = 1:max_session
        all_values{s} = all_values{s}*length(all_values{s}); %#ok<AGROW>
    end
end


% Get the means and stderrs
means = zeros(1,max_session);
stderrs = zeros(1,max_session);
for s = 1:max_session
    means(s) = mean(all_values{s}(~isnan(all_values{s})));
    stderrs(s) = std(all_values{s}(~isnan(all_values{s}))) / sqrt(length(all_values{s}(~isnan(all_values{s}))));
    if ~isempty(all_ages)
        age_means(s) = mean(all_ages{s}(~isnan(all_values{s})));
        age_stderrs(s) = std(all_ages{s}(~isnan(all_values{s}))) / sqrt(length(all_values{s}(~isnan(all_values{s}))));
    end
end

% Normalize here
if num_boots > 0
    if length(means) > max_norm_session
        means = means(1:max_norm_session) - norms;
        
        % Comment this in to produce the norms instead
%         means = norms;
    else
        means = means - norms(1:length(means));

        % Comment this in to produce the norms instead
%         means = norms(1:length(means));
    end
end


% Save the means and stderrs
if ~isempty(values_file)
    fprintf(fid,'Averages');
    for m = 1:length(means)
        fprintf(fid,['\t' num2str(means(m))]);
    end
    fprintf(fid,'\n');
    fclose(fid);

    fprintf(fid,'Stderr');
    for e = 1:length(stderrs)
        fprintf(fid,['\t' num2str(stderrs(e))]);
    end
    fprintf(fid,'\n');
    fclose(fid);
end

if display_type > 0
    
    % Plot the means
    if display_type == 1 || display_type == 3
        scatter(1:max_session,means,'filled');
    end
    if display_type == 2
        errorbar(1:max_session,means,stderrs,'o');
    end
    
    % Could be a histogram for one values
    if display_type == 4 
        if length(all_values) > 1
            error('Histogram only intended for single value stats.')
        end
        hist(all_values{1});
    else

        % Cut out too few data points (without means)
        %   that we might have plotted earlier.
        curr_ax = axis;
        axis_max = min(16+1, max_session+1);
        axis([0 axis_max curr_ax(3) curr_ax(4)]);
%         axis([0 max_session+1 curr_ax(3) curr_ax(4)]);
        if num_boots > 0

            % Add a line
            for i = 1:max_session
                line([i-1 i-.75],[0 0]); 
                line([i-.25 i],[0 0]); 
            end        
        end
    end
    
    % And save
    if ~isempty(image_file)
        saveas(gcf,image_file);
    end
end

% Helper to get only the good sessions.
%   * Syllable must be above baseline
%   * There must be a certain number of minimum occurrences
%   * The session must be parsed to a certain extent
%   * The session must be a certain percent babbling
function ind = getGoodSessions(s, syl_above_baseline, results, min_syl_freq, min_perc_babble, min_perc_parse)

% Return the filtered sessions
ind = find(syl_above_baseline(s,:) > 0 & ...
    squeeze(results.syl_freq(1,s,:) > min_syl_freq)' & ...
    squeeze(results.perc_babbling > min_perc_babble) & ...
    squeeze(results.perc_parsed > min_perc_parse));

