%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% File: plotMultipleTrends.m
%
% This creates plots across many children.
%
% This expects all of the analyses for the
%   given children to have been precomputed
%   using analyzeChild and saved into
%   analysis_directory with the suffix
%   '_analysis_results.mat' following
%   the child name.
%
% Author: Doug Bemis
% Date: 12/7/11
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [means stderrs child_means unused_data age_means age_stderrs child_age_means] = plotMultipleTrends(children, analysis_directory, trend_type, ...
    alignment, min_child_sessions, syl_baseline, tr_baseline, ignore_prior_syllables, min_syl_freq, min_sess_syl, min_babbling, ...
    min_parsed, min_age, num_boots, image_file, values_file, display_type, c_image_file_suf, c_val_file_suf, c_disp_type,...
    display_boots, c_boots_image_file_suf, use_known_transitions)


% Get the values for each child
unused_data = [];
child_means = {};
child_age_means = {};
max_session = 100000;
num_children = 0;
included_children = {};
for c = 1:length(children)
    
    % Load the values
    child = children{c};
    disp(['Calculating for ' child '...']);
    load([analysis_directory '/' child '_analysis_results.mat']);

    % Make a new figure if we're plotting individuals
    if c_disp_type > 0 || display_boots
        figure;
    end
    
    % Make the save files, if asked
    if ~isempty(c_image_file_suf)
        c_image_file = [child '_' c_image_file_suf];
    else
        c_image_file = '';
    end
    if ~isempty(c_val_file_suf)
        c_val_file = [child '_' c_val_file_suf];
    else
        c_val_file = '';
    end
    if ~isempty(c_boots_image_file_suf)
        c_boots_image_file = [child '_' c_boots_image_file_suf];
    else
        c_boots_image_file = '';
    end        
    if ~exist('use_known_transitions','var')
        use_known_transitions = 0;
    end
    
    % Do the individual plot
    [c_means c_stderrs c_all_values c_age_means c_age_stderrs c_norms c_norm_errs] = plotTrends(results ,trend_type, alignment,...
        syl_baseline, tr_baseline, ignore_prior_syllables,min_syl_freq, ...
        min_sess_syl, min_babbling, min_parsed, min_age, num_boots, ...
        c_image_file, c_val_file,c_disp_type, babbles, files, ...
        display_boots, c_boots_image_file, use_known_transitions); 

%      c_means = c_means*length(c_all_values);
    % Store the calculated data
    if length(c_means) >= min_child_sessions
        num_children = num_children+1;
        child_means{num_children} = c_means; %#ok<AGROW>
        child_n_means{num_children} = c_norms; %#ok<AGROW>
        if isempty(c_age_means)
            child_age_means{num_children} = zeros(size(c_means)); %#ok<AGROW>
        else
            child_age_means{num_children} = c_age_means; %#ok<AGROW>
        end
        included_children{num_children} = child; %#ok<AGROW>

        % Might want to plot all points
        if display_type == 1
            scatter(1:length(child_means{num_children}),child_means{num_children});
        end

        % See if we have to lower the max
        if length(child_means{num_children}) < max_session
            max_session = length(child_means{num_children});
        end
        
    % If not good enough, store as unused
    else
        unused_data{end+1} = {child, c_means}; %#ok<AGROW>
    end
end
    
% Make all graphs the same length
max_session = min_child_sessions;

% Reverse this so time goes forward
if alignment == 3
    for c = 1:num_children
        child_means{c} = child_means{c}(max_session:-1:1); %#ok<AGROW>
        child_age_means{c} = child_age_means{c}(max_session:-1:1); %#ok<AGROW>
        if num_boots > 0
            child_n_means{c} = child_n_means{c}(max_session:-1:1); %#ok<AGROW>
        end
    end 
end

% Reorganize and calculate
all_means = zeros(num_children,max_session);
all_age_means = zeros(num_children,max_session);
for c = 1:num_children
    all_means(c,:) = child_means{c}(1:max_session);
    all_age_means(c,:) = child_age_means{c}(1:max_session);
    if num_boots > 0
        all_n_means(c,:) = child_n_means{c}(1:max_session);    
    end
end

% Get the means to plt
if num_children > 1
    means = mean(all_means);
    stderrs = std(all_means) / sqrt(num_children);
    age_means = mean(all_age_means);
    age_stderrs = std(all_age_means) / sqrt(num_children);
    if num_boots > 0
        n_means = mean(all_n_means);
        n_stderrs = std(all_n_means) / sqrt(num_children);
    end
    
% Not much to do if only one child
else
    means = all_means;
    if num_boots > 0
        n_means = all_n_means;
    end
    stderrs = [];
end

% And show
if display_type > 0

    % Might need a new one
    if c_disp_type > 0 || display_boots
        figure;
    end
    
    % Plot the means
    chance_level = 0;
    if display_type == 1 || display_type == 3 || num_children == 1
%         scatter(1:max_session,means,'filled');
        plot(1:max_session,means);
    end
    if display_type == 2 && num_children > 1
        errorbar(1:max_session,means,stderrs,'o');
    end
    if display_type == 4 && num_children > 1
        errorbar(1:max_session,means+n_means,stderrs,'o');
        hold on;
        plot(1:max_session,n_means,'r');
        hold off;
    end
    if display_type == 5 && num_children > 1
        chance_level = 1;
        errorbar(1:max_session,(means+n_means)./n_means,stderrs,'o');
    end

    % Cut out too few data points (without means)
    %   that we might have plotted earlier.
    curr_ax = axis;
    axis([0 max_session+1 curr_ax(3) curr_ax(4)]);
        
    % Add a line if useful
    if num_boots > 0
        for i = 1:max_session
            line([i-1 i-.75],[chance_level chance_level]); 
            line([i-.25 i],[chance_level chance_level]); 
        end        
    end
    
    % And save
    if ~isempty(image_file)
        saveas(gcf,image_file);
    end
end


% Save if we want to
if ~isempty(values_file)
    fid = fopen(values_file,'w');

    % Print the header
    fprintf(fid,'Child');
    for s = 1:max_session
        fprintf(fid,['\t' num2str(s)]);
    end
    fprintf(fid,'\n');

    % The values
    for c = 1:num_children
        fprintf(fid,included_children{c});
        for s = 1:max_session
            fprintf(fid,['\t' num2str(child_means{c}(s))]); 
        end
        fprintf(fid,'\n');
    end
    
    % And the summaries
    fprintf(fid,'Averages');
    for s = 1:max_session
        fprintf(fid,['\t' num2str(means(s))]);
    end
    fprintf(fid,'\n');
    if ~isempty(stderrs)
        fprintf(fid,'Stderrs');
        for s = 1:max_session
            fprintf(fid,['\t' num2str(stderrs(s))]);
        end
        fprintf(fid,'\n');
    end
    fclose(fid);
end

